module List where

open import Nat
open import Equality
open import Ordering
open import Decidable


data List (A : Set) : Set where
  nil : List A
  _::_ : A -> List A -> List A
{-# BUILTIN LIST List #-}
infixr 9 _::_

-- Calculate the length of a list.
length : {A : Set} -> List A -> ℕ
length nil = 0
length (x :: xs) = succ (length xs)

-- Append an element to the end of the list.
_:>_ : {A : Set} -> List A -> A -> List A
nil :> y = y :: nil
(x :: xs) :> y = x :: (xs :> y)

-- Reverse a list.
reverse : {A : Set} -> List A -> List A
reverse nil = nil
reverse (x :: xs) = reverse xs :> x


-- Define an "append" function that concatenates two lists. 
_++_ : {A : Set} -> List A -> List A -> List A
nil ++ ys = ys
(x :: xs) ++ ys = x :: (xs ++ ys)
infixl 7 _++_

lemma-append-nil : ∀ {A : Set} -> (xs : List A) -> xs ++ nil == xs
lemma-append-nil nil = refl
lemma-append-nil (x :: xs) = context (\xs -> x :: xs)  (lemma-append-nil xs)

-- Append is associative.
lemma-append-assoc : ∀ {A : Set} -> (xs ys zs : List A) -> (xs ++ ys) ++ zs == xs ++ (ys ++ zs)
lemma-append-assoc nil ys zs = refl
lemma-append-assoc (x :: xs) ys zs = context (\xs -> x :: xs) (lemma-append-assoc xs ys zs)

-- "++" interacts with ":>" as expected.
lemma-++-:> : ∀ {A} (xs ys : List A) (y : A) -> xs ++ (y :: ys) == (xs :> y) ++ ys
lemma-++-:> nil ys y = refl
lemma-++-:> (x :: xs) ys y = context (λ xs → x :: xs) (lemma-++-:> xs ys y)


map : {A B : Set} -> (A -> B) -> List A -> List B
map f nil = nil
map f (x :: xs) = f x :: map f xs


lemma-map-append : ∀ {A B : Set} -> (f : A -> B) -> (xs ys : List A) -> map f (xs ++ ys) == map f xs ++ map f ys
lemma-map-append f nil ys = refl
lemma-map-append f (x :: xs) ys = context (\xs -> f x :: xs) (lemma-map-append f xs ys)


reverse-aux :  {A : Set} -> List A -> List A -> List A
reverse-aux nil ys = ys
reverse-aux (x :: xs) ys = reverse-aux xs (x :: ys)

reverse' : {A : Set} -> List A -> List A
reverse' {A} xs = reverse-aux {A} xs nil


lemma-reverse-aux-length : ∀ {A : Set} -> (xs ys : List A) -> length (reverse-aux xs ys) == length xs + length ys
lemma-reverse-aux-length nil ys = refl
lemma-reverse-aux-length (x :: xs) ys =
  equational length (reverse-aux xs (x :: ys))
    by lemma-reverse-aux-length xs (x :: ys)
  equals length xs + succ (length ys)
   by symm (lemma-add-succ (length xs) (length ys))
  equals succ (length xs + length ys)

thm-reverse'-length : ∀ {A : Set} -> (xs : List A) -> length xs == length (reverse' xs)
thm-reverse'-length xs =
  equational length xs
    by lemma-add-m-zero (length xs)
   equals length xs + 0
    by symm (lemma-reverse-aux-length xs nil)
   equals length (reverse' xs)


lemma-reverse-aux : ∀ {A : Set} -> (xs ys : List A) -> reverse-aux xs ys == reverse xs ++ ys
lemma-reverse-aux nil ys = refl
lemma-reverse-aux (x :: xs) ys =
  equational reverse-aux (x :: xs) ys
          by definition   -- "definition" is a synonym for "refl", defined in Equality.agda
      equals reverse-aux xs (x :: ys)
          by lemma-reverse-aux xs (x :: ys)
      equals reverse xs ++ (x :: ys)
          by lemma-++-:> (reverse xs) ys x
      equals (reverse xs :> x) ++ ys
          by definition
      equals reverse (x :: xs) ++ ys

thm-reverse'-is-reverse : ∀ {A : Set} -> (xs : List A) -> reverse' xs == reverse xs
thm-reverse'-is-reverse xs =
  equational reverse' xs
          by definition
      equals reverse-aux xs nil
          by lemma-reverse-aux xs nil
      equals reverse xs ++ nil
          by lemma-append-nil (reverse xs)
      equals reverse xs

data List-∀ {A : Set} : (A -> Set) -> List A -> Set where
  base-nil : ∀ P -> List-∀ P nil
  step-cons : ∀ P x xs -> P x -> List-∀ P xs -> List-∀ P (x :: xs)


elemDecide : ∀ {A : Set} -> (P : A -> Set) -> (∀ x -> Decidable (P x)) -> (∀ xs -> Decidable (List-∀ P xs))
elemDecide P decP nil = yes (base-nil P)
elemDecide P decP (x :: xs) with decP x | elemDecide P decP xs
elemDecide P decP (x :: xs) | yes Px | yes Pxs = yes (step-cons P x xs Px Pxs)
elemDecide P decP (x :: xs) | yes Px | no nPxs = no λ { (step-cons .P .x .xs Px' Pxs) → nPxs Pxs}
elemDecide P decP (x :: xs) | no nPx | hyp2 = no (λ { (step-cons .P .x .xs Px Pxs) → nPx Px})
